/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright (c) 2015 Adobe Systems Incorporated. All rights reserved.
 *
 * NOTICE:    All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.    The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

/*jslint vars: true, plusplus: true, devel: true, nomen: true, indent: 4, bitwise: true, node: true, expr: true*/
/*global describe: true, _: true, expect: true, window: true, WebSocket: true, it: true, afterEach */
"use strict";

var AssetExporter = require("../lib/asset-exporter"),
    PreviewCache = require("../lib/preview-cache"),
    Renderer = require('generator-assets/lib/renderer'),
    Q = require("q"),
    fs = require("fs"),
    sinon = require("sinon");

describe('AssetExporter', function() {
    var sandbox = sinon.sandbox.create();

    afterEach(function () {
        sandbox.restore();
    });

    var FakeRenderer = {
        render: function() { return Q.resolve({ path: "/tmp/123" }); }
    };

    it("should use an SVGRenderer for SVG files", function() {
        sandbox.stub(Renderer, "createSVGRenderer").returns(FakeRenderer);
        sandbox.spy(FakeRenderer, "render");

        var options = {
            component: {
                extension: "svg",
                layer: {
                    document: {}
                }
            }
        };

        var promise = AssetExporter.exportAsset(options);
        return expect(promise).to.eventually.be.fulfilled
            .then(function() {
                expect(Renderer.createSVGRenderer).to.have.been.calledOnce;
                expect(FakeRenderer.render).to.have.been.calledOnce;
            });
    });

    it("should use a PixmapRenderer for a bitmap file format", function() {
        sandbox.stub(Renderer, "createPixmapRenderer").returns(FakeRenderer);
        sandbox.spy(FakeRenderer, "render");

        var options = {
            component: {
                extension: "jpg",
                layer: {
                    document: {}
                }
            }
        };

        var promise = AssetExporter.exportAsset(options);
        return expect(promise).to.eventually.be.fulfilled
            .then(function() {
                expect(Renderer.createPixmapRenderer).to.have.been.calledOnce;
                expect(FakeRenderer.render).to.have.been.calledOnce;
            });
    });

    it("should write cached preview data for 1x component", function() {
        var options = {
                component: {
                    path: "test.jpg",
                    isPreview: true,    // 1x image
                    previewId: 123
                }
            },
            cachedPreview = {
                buffer: new Buffer([0,1,2,3,4,5,6,7])
            };

        sandbox.stub(PreviewCache, "get").returns(cachedPreview);
        sandbox.stub(fs, "writeFile").returns(Q.resolve());
        sandbox.stub(Q, "nfcall").callsArg(0).returns(Q.resolve());

        var promise = AssetExporter.exportAsset(options);
        expect(PreviewCache.get).to.have.been.calledOnce;
        expect(PreviewCache.get.firstCall.args[0]).to.equal(options.component.previewId);
        expect(Q.nfcall).to.have.been.calledOnce;
        expect(Q.nfcall.firstCall.args[1]).to.equal(options.component.path);
        expect(Q.nfcall.firstCall.args[2]).to.equal(cachedPreview.buffer);
    });
});
